local PANEL = {};

// link to class
function PANEL:LinkToClass( cls )
	self.LinkedClass = cls;
	
end

// children hide.
function PANEL:HideChildren( )
	// for every one of our children hide and call hide.
	local child;
	for _, child in pairs( self.Children ) do
		// hide it
		child:SetVisible( false );
		
		// hide it's children
		child:HideChildren();
		
	end
	
end

// show children
function PANEL:ShowChildren( )
	// for every one of our children hide and call hide.
	local child;
	for _, child in pairs( self.Children ) do
		// hide it
		child:SetVisible( true );
		
		// hide it's children
		if( child.Expanded ) then
			child:ShowChildren();
			
		end
		
	end
	
end

// remove children.
function PANEL:RemoveChildren( )
	// for every one of our children hide and call hide.
	local child;
	for _, child in pairs( self.Children ) do
		// hide it
		child:Remove();
		
	end
	
	// wipe.
	self.Children = {};
	
end

// get node
function PANEL:GetNode( idx )
	return self.Children[idx];

end

// get node count
function PANEL:GetNodeCount(  )
	return table.getn( self.Children );

end



// expand callback
function PANEL:ExpandPressed( )
	// pass along to the treeview
	if( self.Treeview ) then
		if( self.Treeview:NodeExpanded( self ) ) then
			return;
			
		end
		
	end

	// toggle expand.
	self.Expanded = !self.Expanded;
	
	// based on new state show/hide.
	if( self.Expanded ) then
		// show children and update expander button.
		self:ShowChildren();
		self.Expander:SetMaterial( AppFramework:Theme().icons['minus_icon'] );
		
	else
		// hide children and update expander button.
		self:HideChildren();
		self.Expander:SetMaterial( AppFramework:Theme().icons['plus_icon'] );
	
	end
	
	// normal
	if( self.Image && ( !self.Selected && !self.Expanded ) ) then
		self:SetIcon( self.Image );
	
	// selected
	elseif( self.SelectedImage && ( self.Selected || self.Expanded ) ) then
		self:SetIcon( self.SelectedImage );
		
	end
	
end


// expand node.
function PANEL:SetExpanded( v )
	// pass along to the treeview
	if( self.Treeview ) then
		if( self.Treeview:NodeExpanded( self ) ) then
			return;
			
		end
		
	end

	// toggle expand.
	self.Expanded = v;
	
	// based on new state show/hide.
	if( self.Expanded ) then
		// show children and update expander button.
		self:ShowChildren();
		self.Expander:SetMaterial( AppFramework:Theme().icons['minus_icon'] );
		
	else
		// hide children and update expander button.
		self:HideChildren();
		self.Expander:SetMaterial( AppFramework:Theme().icons['plus_icon'] );
	
	end
	
	// normal
	if( self.Image && ( !self.Selected && !self.Expanded ) ) then
		self:SetIcon( self.Image );
	
	// selected
	elseif( self.SelectedImage && ( self.Selected || self.Expanded ) ) then
		self:SetIcon( self.SelectedImage );
		
	end
	
end


// init
function PANEL:Init( )
	// linked class
	self.LinkedClass = nil;

	// expander.
	self.Expander = CtlIconButton_class:create( self );
	self.Expander:SetMaterial( AppFramework:Theme().icons['plus_icon'] );
	self.Expander:AddEvent( "OnClick", "ExpandPressed", self );
	
	// label
	self.Label = vgui.Create( "label", self );
	self.Label:SetMouseInputEnabled( false );
	
	// icon
	self.Icon = nil;
	self.Text = "";
	
	// the treeview we're bound to.
	self.Treeview = nil;
	
	// child nodes
	self.Children = {};
	self.Parent = nil;
	
	// expanded state.
	self.Expanded = false;
	self.ForceExpandable = false;
	
	// selected?
	self.Selected = false;
	
	// images
	self.Image = 0;
	self.SelectedImage = 0;
	
	// is end cap?
	self.DrawList = false;
	self.EndList = false;
	self.Indent = 0;
	
end

// set icon
function PANEL:SetIcon( mat )
	// create
	if( !self.Icon ) then
		self.Icon = vgui.Create( "ctl_icon", self );
		self.Icon:SetMouseInputEnabled( false );
		
	end
	
	// full texture.
	if( type( mat ) == "string" ) then
		self.Icon:SetMaterial( mat );
		
	// imagelist?
	elseif( type( mat ) == "number" && self.Treeview && self.Treeview.ImageList ) then
		local texture = self.Treeview.ImageList:Get( mat );
		if( texture ) then
			self.Icon.Material = texture;
			
		end
	
	end
	
end

// set images.
function PANEL:SetImages( normal, selected )
	self.Image = normal;
	self.SelectedImage = selected;
	
	// normal
	if( self.Image && ( !self.Selected && !self.Expanded ) ) then
		self:SetIcon( self.Image );
	
	// selected
	elseif( self.SelectedImage && ( self.Selected || self.Expanded ) ) then
		self:SetIcon( self.SelectedImage );
		
	end
	
end

// set text
function PANEL:SetText( txt )
	self.Label:SetText( txt );
	self.Text = txt;
	
end

function PANEL:GetText( )
	return self.Text;
	
end

// get/set selected
function PANEL:GetSelected( )
	return self.Selected;
	
end

// set force expandable
function PANEL:SetForceExpandable( v )
	self.ForceExpandable = v;
	
end

// set selected
function PANEL:SetSelected( v )
	self.Selected = v;
	
	// normal
	if( self.Image && ( !self.Selected && !self.Expanded ) ) then
		self:SetIcon( self.Image );
	
	// selected
	elseif( self.SelectedImage && ( self.Selected || self.Expanded ) ) then
		self:SetIcon( self.SelectedImage );
		
	end
	
end


// select our node.
function PANEL:OnMousePressed( mc )
	// if we have a treeview, deselect previous node, and select this one.
	if( self.Treeview ) then
		// select this node.
		self.Treeview:SetSelectedNode( self );
		
		// pass nodeclick on.
		if( self.Treeview.LinkedClass ) then
			self.Treeview.LinkedClass:CallEvent( "OnNodeClicked", self, mc );
			
		end
		
	end

end

// perform layout
function PANEL:PerformLayout( )
	// calculate offset from indent level/
	local ofsx = self.Indent * 16;
	
	// do we have children?
	if( table.getn( self.Children ) > 0 || self.ForceExpandable ) then
		// position the expander.
		self.Expander:SetSize( 16, 16 );
		self.Expander:SetPos( ofsx, 0 );
		if( !self.Expander:IsVisible() ) then self.Expander:Show(); end
	
	// hide.
	else
		// hide expander.
		self.Expander:Hide();
		
	end
	
	// shift
	ofsx = ofsx + 18;
	
	// position the icon if we have one.
	if( self.Icon ) then
		// position icon
		self.Icon:SetSize( 16, 16 );
		self.Icon:SetPos( ofsx, 0 );
		
		// shift offset.
		ofsx = ofsx + 18;
		
	end
	
	// position label.
	self.Label:SizeToContents();
	self.Label:SetPos( ofsx, 8 - self.Label:GetTall() * 0.5 );
	
	// size myself
	self:SetSize( ofsx + self.Label:GetWide() + 16, 16 );
	
end


// scheme settings.
function PANEL:ApplySchemeSettings( )
	self.Label:SetFont( AppFramework:Theme().fonts['text'] );
	self.Label:SetFGColor( self.Color or AppFramework:Theme().colors['text'] );
	
end

// set color
function PANEL:SetTextColor( c )
	self.Color = c;
	self.Label:SetFGColor( self.Color );
	
end


// paint.
function PANEL:Paint( )
	// draw selection box.
	if( self:GetSelected() ) then
		// get select size.
		local x, y = self.Label:GetPos();
		local w, h = self.Label:GetSize();
		
		// draw
		local color = AppFramework:Theme().colors['selection'];
		surface.SetDrawColor(
			color.r,
			color.g,
			color.b,
			color.a
		);
		surface.DrawRect(
			x, y,
			w, h
		);
		
	end
	
	// draw connection lines.
	if( self.DrawList ) then
		// draw outside line.
		local color = AppFramework:Theme().colors['border_dark'];
		surface.SetDrawColor(
			color.r,
			color.g,
			color.b,
			color.a
		);

		// icon offset.
		local ofsx = self.Indent * 16 - 8;
		
		// draw horizontal line.
		surface.DrawLine(
			ofsx, 8,
			ofsx + 16, 8
		);
		
		// draw half side
		if( self.EndList && !( self.Expanded && table.getn( self.Children ) ) ) then
			surface.DrawLine(
				ofsx, 0,
				ofsx, 8
			);
			
		// draw full side.
		else
			surface.DrawLine(
				ofsx, 0,
				ofsx, 16
			);
		end
		
		// draw parent.
		local parent = self.Parent;
		while( parent ) do
			// shrink offset.
			ofsx = ofsx - 16;
			
			// draw.
			surface.DrawLine(
				ofsx, 0,
				ofsx, 16
			);
			
			// walk
			parent = parent.Parent;
			
		end

	end
	
	//
	return true;
	
end


// do tree layout
function PANEL:DoTreeLayout( indent, ofsy, widest )
	// calculate my position.
	self:SetPos( 0, ofsy );
	ofsy = ofsy + 15;
	
	// set my indent level.
	self.Indent = indent;
	
	// lets go ahead and size things up.
	self:PerformLayout();
	
	// how big are we?
	if( self:GetWide() > widest ) then widest = self:GetWide(); end
	
	// am I expanded? no children?
	if( !self.Expanded || !table.getn( self.Children ) ) then return indent, ofsy, widest; end
	
	// increase indent level
	indent = indent + 1;
	
	// calculate our children.
	local child, lastchild;
	for _, child in pairs( self.Children ) do
		// calc
		_, ofsy, widest = child:DoTreeLayout( indent, ofsy, widest );
		
		// draw list
		child.DrawList = true;
		
		// store last child.
		lastchild = child;
		
	end
	
	// set lastchild to endlist
	if( lastchild ) then lastchild.EndList = true; end
	
	// return ofs
	return indent, ofsy, widest;
	
end


// set treeview.
function PANEL:SetTreeview( treeview )
	self.Treeview = treeview;
	
end

// add node
function PANEL:AddNode( node )
	// add to children
	table.insert( self.Children, node );
	
	// set parent
	node.Parent = self;
	
	// are we expanded? make it visible.
	node:SetVisible( self.Expanded );
	
end

// add node.
function PANEL:CreateNode( )
	if( !self.Treeview ) then return; end
	
	// add.
	local node = vgui.Create( "ctl_treenode", self.Treeview.Canvas:GetCanvas() );
	node:SetTreeview( self.Treeview );
	
	// set parent
	node.Parent = self;
	
	// are we expanded? make it visible.
	node:SetVisible( self.Expanded );
	
	// add
	table.insert( self.Children, node );
	
	//
	return node;
	
end

// register
vgui.Register( "ctl_treenode", PANEL, "Panel" );

